import os, hashlib
from django.conf import settings
from webapp.common import create_thumbnail, update_tgl
from webapp.models import koleksi as m_koleksi

# Path upload
RELPATH = 'assets/upload/surat'
ABSPATH = os.path.join(settings.BASE_DIR, RELPATH)
os.makedirs(ABSPATH, exist_ok=True)


def deploy(request, pk_surat, files):
    """
    Simpan file baru (upload lampiran baru) ke Suratf.
    """
    for file in files:
        random_ = update_tgl()['tgldetik']

        suratf = m_koleksi.Suratf.objects.create(
            pk_surat=m_koleksi.Surat.objects.get(pk=pk_surat),
            name=file.name,
            size=file.size,
            _type=file.content_type,
            is_deleted=False  # Aktif
        )

        # Generate path
        infix = f"{suratf.created.year:04}{suratf.created.month:02}{suratf.created.day:02}"
        suffix = f"{random_}_{hashlib.md5(file.name.encode()).hexdigest()}_{suratf.pk}{os.path.splitext(file.name)[1]}"
        full_path = os.path.join(ABSPATH, infix, suffix)

        os.makedirs(os.path.join(ABSPATH, infix), exist_ok=True)

        # Upload ke disk
        with open(full_path, 'xb+') as buffer:
            for chunk in file.chunks():
                buffer.write(chunk)

        suratf.path = os.path.join(infix, suffix).replace("\\", "/")
        suratf.save()

        create_thumbnail(suratf.path, ABSPATH)


def destroy_one(pk):
    """
    Soft delete satu lampiran.
    """
    m_koleksi.Suratf.objects.filter(pk=pk).update(is_deleted=True)


def destroy_many(pk_surat):
    """
    Soft delete semua lampiran untuk surat tertentu.
    """
    m_koleksi.Suratf.objects.filter(pk_surat=pk_surat).update(is_deleted=True)


def destroy_one_permanent(pk):
    """
    Hard delete lampiran (hapus file fisik & DB row).
    """
    suratf = m_koleksi.Suratf.objects.filter(pk=pk).first()
    if suratf:
        try:
            full_path = os.path.join(ABSPATH, suratf.path)
            if os.path.exists(full_path):
                os.remove(full_path)
        except Exception as e:
            print('❌ Error hapus file fisik:', e)

        suratf.delete()


def save_surat_version(surat_obj, user_id):
    """
    Simpan metadata surat & lampiran aktif ke tabel SuratVersion & SuratVersionAttachment.
    """
    version = m_koleksi.SuratVersion.objects.create(
        pk_surat=surat_obj.pk,
        nomor=surat_obj.nomor,
        tanggal=surat_obj.tanggal,
        tanggal_masuk=surat_obj.tanggal,
        pengirim=surat_obj.pengirim,
        pengirim_nama=surat_obj.pengirim_nama,
        pengirim_instansi=surat_obj.pengirim_instansi,
        penerima=surat_obj.penerima,
        penerima_nama=surat_obj.penerima_nama,
        penerima_instansi=surat_obj.penerima_instansi,
        sifat=surat_obj.sifat,
        jenis=surat_obj.jenis,
        perihal=surat_obj.perihal,
        status_surat=surat_obj.status_surat,
        user_input=user_id
    )

    attachments = m_koleksi.Suratf.objects.filter(pk_surat=surat_obj.pk, is_deleted=False)
    for att in attachments:
        m_koleksi.SuratVersionAttachment.objects.create(
            pk_version=version,
            name=att.name,
            path=att.path,
            _type=att._type,
            size=att.size
        )


def rollback_surat(pk_surat, version_id, user_id):
    """
    Rollback surat ke versi tertentu.
    Backup versi aktif dulu ke history, lalu restore metadata & file.
    """
    surat = m_koleksi.Surat.objects.get(pk=pk_surat)
    version = m_koleksi.SuratVersion.objects.get(pk=version_id)

    # Backup versi aktif
    save_surat_version(surat, user_id)

    # Update metadata surat
    surat.nomor = version.nomor
    surat.tanggal = version.tanggal
    surat.tanggal_masuk = version.tanggal_masuk
    surat.pengirim = version.pengirim
    surat.pengirim_nama = version.pengirim_nama
    surat.pengirim_instansi = version.pengirim_instansi
    surat.penerima = version.penerima
    surat.penerima_nama = version.penerima_nama
    surat.penerima_instansi = version.penerima_instansi
    surat.sifat = version.sifat
    surat.jenis = version.jenis
    surat.perihal = version.perihal
    surat.status_surat = version.status_surat
    surat.save()

    # Soft delete lampiran lama
    m_koleksi.Suratf.objects.filter(pk_surat=pk_surat).update(is_deleted=True)

    # Restore lampiran dari version
    for v_att in m_koleksi.SuratVersionAttachment.objects.filter(pk_version=version.pk):
        m_koleksi.Suratf.objects.create(
            pk_surat=surat,
            name=v_att.name,
            path=v_att.path,
            _type=v_att._type,
            size=v_att.size,
            is_deleted=False
        )
