# ========================
# Python Standard Library
# ========================
import os
import uuid
import tempfile
from io import BytesIO

# ========================
# Third-Party Libraries
# ========================
from docx import Document
from PyPDF2 import PdfReader
from PIL import Image
from pdf2image import convert_from_bytes, convert_from_path
from pdf2docx import Converter
import fitz  # PyMuPDF

# ========================
# Django Core
# ========================
from django.conf import settings
from django.core.files.base import ContentFile
from django.core.files.storage import default_storage
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.db import connection
from django.db.models import Q, Value, F, CharField, TextField, IntegerField
from django.db.models.functions import Cast
from django.http import (
    HttpResponse, 
    HttpResponseRedirect, 
    JsonResponse, 
    Http404
)
from django.shortcuts import render, get_object_or_404
from django.utils.text import slugify
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST
from django.contrib import messages

# ========================
# Local Apps
# ========================
from webapp.common import dump, currency, tgl_indo
from webapp.models import koleksi as m_koleksi, master as m_master
from webapp.models.koleksi import (
    ContentVersion,
    ContentVersionAttachment,
    Contents,
    ContentTag,
    Tag,
)
from webapp.models.master import Users
from webapp.utils import upload_to_drive  # 🔄 Perubahan: upload ke Google Drive


# =====================================
# Upload PDF ke Google Drive
# =====================================
def upload_pdf_to_drive(request):
    """
    Upload PDF ke Google Drive dan kembalikan link.
    """
    if request.method == 'POST':
        file = request.FILES.get('file')
        if file:
            try:
                link = upload_to_drive(file, file.name)
                return JsonResponse({
                    'status': 'success',
                    'file_name': file.name,
                    'web_link': 'https://drive.google.com/drive/folders/19pH19iCJ8hNr9OF0TwyWwMiKeR-_iecS'
                })
            except Exception as e:
                return JsonResponse({'status': 'error', 'message': str(e)})
        else:
            return JsonResponse({'status': 'error', 'message': 'File tidak ditemukan.'})

    return JsonResponse({'status': 'error', 'message': 'Metode request harus POST'})


# =====================================
# Konversi PDF ke Gambar
# =====================================
@csrf_exempt
@require_POST
def convert_to_images(request):
    """
    Konversi PDF ke beberapa gambar (PNG).
    """
    pdf_file = request.FILES.get('file')
    if not pdf_file:
        return JsonResponse({'status': 'error', 'message': 'No PDF file provided'}, status=400)

    unique_id = uuid.uuid4().hex[:8]
    base_name = slugify(os.path.splitext(pdf_file.name)[0])
    filename = f"{base_name}-{unique_id}.pdf"
    pdf_path = os.path.join(settings.MEDIA_ROOT, 'converted_pdfs', filename)

    os.makedirs(os.path.dirname(pdf_path), exist_ok=True)

    # Simpan PDF
    with default_storage.open(pdf_path, 'wb+') as destination:
        for chunk in pdf_file.chunks():
            destination.write(chunk)

    # Konversi ke gambar
    doc = fitz.open(pdf_path)
    image_urls = []
    image_folder = os.path.join(settings.MEDIA_ROOT, 'converted_images', f"{base_name}-{unique_id}")
    os.makedirs(image_folder, exist_ok=True)

    for i in range(len(doc)):
        page = doc.load_page(i)
        pix = page.get_pixmap(matrix=fitz.Matrix(2, 2))  # Resolusi lebih tinggi
        image_filename = f"page_{i+1}.png"
        image_path = os.path.join(image_folder, image_filename)
        pix.save(image_path)

        image_url = f"{settings.MEDIA_URL}converted_images/{base_name}-{unique_id}/{image_filename}"
        image_urls.append(image_url)

    doc.close()
    return JsonResponse({'status': 'success', 'images': image_urls})


# =====================================
# Konversi PDF ke Word
# =====================================
@csrf_exempt
@require_POST
def convert_to_word(request):
    """
    Konversi PDF ke file Word (.docx).
    """
    file = request.FILES.get('file')
    if not file or not file.name.endswith('.pdf'):
        return HttpResponse("File tidak valid. Harap unggah file PDF.", status=400)

    with tempfile.NamedTemporaryFile(delete=False, suffix=".pdf") as tmp_pdf:
        tmp_pdf.write(file.read())
        tmp_pdf_path = tmp_pdf.name

    tmp_docx_path = tmp_pdf_path.replace('.pdf', '.docx')
    cv = Converter(tmp_pdf_path)
    cv.convert(tmp_docx_path, start=0, end=None)
    cv.close()

    with open(tmp_docx_path, 'rb') as docx_file:
        response = HttpResponse(
            docx_file.read(),
            content_type='application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        )
        response['Content-Disposition'] = f'attachment; filename="{os.path.basename(tmp_docx_path)}"'

    os.remove(tmp_pdf_path)
    os.remove(tmp_docx_path)
    return response


# =====================================
# Daftar Konten (List)
# =====================================
def page_list(request, pk=None):
    """
    Tampilkan daftar konten dengan filter, sorting, dan pagination.
    """
    my_filter = {}
    f = Q()

    term = request.GET.get('term', '').strip()
    search_skpd = request.GET.get('pk_skpd_dg')
    tahun = request.GET.get('tahun')
    disp = request.GET.get('disp')
    item_jml = request.GET.get('item_jml')
    sorting = request.GET.get('sorting')
    reset = request.GET.get('reset', 0)
    page = int(request.GET.get('page', 1) or 1)

    if pk and pk != 1:
        request.session['search_kategori'] = pk

    if search_skpd:
        request.session['search_skpd'] = search_skpd

    if tahun and tahun.isnumeric():
        request.session['tahun'] = int(tahun)

    if disp:
        request.session['display'] = 'list' if disp == 'list' else 'col'

    if item_jml and item_jml.isnumeric():
        request.session['show'] = int(item_jml)

    if sorting:
        request.session['sorting'] = sorting

    if term:
        request.session['term'] = term

    if reset == '1':
        for key in ['term', 'search_skpd', 'search_kategori', 'tahun']:
            request.session.pop(key, None)

    if request.session.get('search_skpd'):
        my_filter['pk_skpd_dg'] = int(request.session['search_skpd'])
    if request.session.get('search_kategori'):
        my_filter['pk_categories'] = int(request.session['search_kategori'])
    if request.session.get('tahun'):
        my_filter['tahun'] = int(request.session['tahun'])

    for key, value in my_filter.items():
        f &= Q(**{key: value})

    if request.session.get('term'):
        term_value = request.session['term']
        fields = [
            field.name
            for field in m_koleksi.Contents._meta.get_fields()
            if isinstance(field, (CharField, TextField)) and not field.is_relation
        ]
        if fields:
            q = Q()
            for field in fields:
                q |= Q(**{f"{field}__icontains": term_value})
            f &= q

    order = '-created' if request.session.get('sorting') == 'desc' else 'created'
    query = m_koleksi.Contents.objects.filter(f).order_by(order).annotate(
        attributess=Cast('attributes', CharField())
    ).defer('attributes')

    item_per_page = request.session.get('show', 6)
    paginator = Paginator(query, item_per_page)
    paging = paginator.get_page(page)

    result = []
    for item in paging:
        item = m_koleksi.Contents._sanitize_result(item)
        # item['attachments'] = m_koleksi.Attachments._read_many(asis=1, pk_contents=item['pk']).count()
        attachments_qs = m_koleksi.Attachments._read_many(asis=1, pk_contents=item['pk'])
        item['attachments'] = attachments_qs.count()
        item['attachments_list'] = [
            {'name': a.name, 'path': a.path} for a in attachments_qs
        ]

        item['thumbnail'] = m_koleksi.Attachments._read_many(asis=1, pk_contents=item['pk'])
        item['pk_skpd_dg'] = m_master.Skpddg._read_many(asis=1, pk=item['pk_skpd_dg']).count()
        result.append(item)

    return render(request, 'fileman/list.html', {
        'pk_categories': pk,
        'categories_list': m_koleksi.Categories._recursive(pk_parent=0, deep=0),
        'skpd_list': m_master.Skpddg.objects.all(),
        'list': result,
        'paging': paging,
        'term_current': request.session.get('term', ''),
        'search_skpd_dg': int(request.session.get('search_skpd')) if str(request.session.get('search_skpd')).isdigit() else '',
        'search_kategori': int(request.session.get('search_kategori')) if str(request.session.get('search_kategori')).isdigit() else '',
        'search_tahun': int(request.session.get('tahun')) if str(request.session.get('tahun')).isdigit() else '',
        'display': request.session.get('display', 'list'),
        'show': item_per_page,
        'sorting': request.session.get('sorting', 'asc'),
    })



def page_item(request, pk):
    """
    Detail satu konten.
    """
    result = m_koleksi.Contents._read_one(asis=1, pk=pk)
    attachments = m_koleksi.Attachments._read_many(asis=1, pk_contents=pk, is_deleted=False)
    status_konten = result.status_konten

    user_pk = request.session.get('pk')
    user_role = request.session.get('role')
    allow = 'yes' if result.user_input == user_pk or user_role in ['ADMIN', 'KOORDINATOR'] else 'no'

    categories, attributes, skpd = None, None, None

    if result.pk_categories:
        categories = m_koleksi.Categories._recursive_reverse(pk=result.pk_categories.pk)
        attr_obj = m_koleksi.Categories._read_one(pk=result.pk_categories.pk)
        attributes = attr_obj['attributes'] if attr_obj else None

        for key in result.attributes:
            if attributes and attributes[key]['type'] == 'money':
                result.attributes[key] = currency(result.attributes[key])
            if attributes and attributes[key]['type'] == 'date':
                result.attributes[key] = tgl_indo(result.attributes[key])

    if result.pk_skpd_dg:
        skpd = m_master.Skpddg._read_one(pk=result.pk_skpd_dg.pk)

    user_yg_input = m_master.Users._read_one(asis=1, pk=result.user_input).usr.upper()
    versions = ContentVersion.objects.filter(pk_contents=pk).order_by('-created')

    # Ambil versi PDF terbaru
    latest_version = versions.first()
    latest_attachment = None
    if latest_version:
        latest_attachment = (
            ContentVersionAttachment.objects.filter(pk_version=latest_version.pk)
            .order_by('-created')
            .first()
        )

    # Sama seperti di page_list → bawa path
    result.latest_pdf = latest_attachment.path if latest_attachment else None

    return render(request, 'fileman/item.html', {
        'item': result,
        'attachments': attachments,
        'categories': categories,
        'attributes': attributes,
        'skpd': skpd,
        'allow': allow,
        'user_yg_input': user_yg_input,
        'status_konten': status_konten,
        'versions': versions,
    })



# # =====================================
# # Riwayat Versi
# # =====================================
# def page_history(request, pk):
#     """
#     Tampilkan riwayat versi konten.
#     """
#     content = m_koleksi.Contents._read_one(pk=pk)

#     with connection.cursor() as cursor:
#         cursor.execute("""
#             SELECT cv.*, u.usr AS user_nama
#             FROM content_versions cv
#             LEFT JOIN users u ON u.pk = cv.user_input
#             WHERE cv.pk_contents = %s
#             ORDER BY cv.created DESC
#         """, [pk])
#         columns = [col[0] for col in cursor.description]
#         versions = [dict(zip(columns, row)) for row in cursor.fetchall()]

#     for v in versions:
#         v['attachments'] = ContentVersionAttachment.objects.filter(pk_version=v['pk'])

#     return render(request, 'fileman/history.html', {
#         'item': content,
#         'versions': versions,
#     })

def page_history(request, pk):
    """
    Tampilkan riwayat versi konten.
    """
    content = m_koleksi.Contents._read_one(pk=pk)

    with connection.cursor() as cursor:
        cursor.execute("""
            SELECT cv.*, u.usr AS user_nama
            FROM content_versions cv
            LEFT JOIN users u ON u.pk = cv.user_input
            WHERE cv.pk_contents = %s
            ORDER BY cv.created DESC
        """, [pk])
        columns = [col[0] for col in cursor.description]
        versions_list = [dict(zip(columns, row)) for row in cursor.fetchall()]

    # Lampirkan attachments
    for v in versions_list:
        v['attachments'] = ContentVersionAttachment.objects.filter(pk_version=v['pk'])

    # ==== PAGINATOR ====
    paginator = Paginator(versions_list, 6)  # 6 per halaman
    page_number = request.GET.get('page')
    versions_page = paginator.get_page(page_number)

    return render(request, 'fileman/history.html', {
        'item': content,
        'versions': versions_page,  # yang dipakai di template: paginated
    })


# # =====================================
# # Detail Tag
# # =====================================
# def page_tag_detail(request, tag_name):
#     """
#     Detail tag dan daftar konten dengan tag tersebut.
#     """
#     tag = get_object_or_404(Tag, name=tag_name.lower())
#     contents = Contents.objects.filter(contenttag__tag=tag).order_by('-created')

#     return render(request, 'fileman/tags_detail.html', {
#         'tag': tag,
#         'contents': contents,
#     })



# =====================================
# Detail Tag dengan Paginasi
# =====================================
def page_tag_detail(request, tag_name):
    """
    Detail tag dan daftar konten dengan tag tersebut.
    """
    tag = get_object_or_404(Tag, name=tag_name.lower())
    contents_list = Contents.objects.filter(contenttag__tag=tag).order_by('-created')

    # Tambahkan Paginator
    paginator = Paginator(contents_list, 6)  # 6 item per halaman
    page = request.GET.get('page')

    try:
        contents = paginator.page(page)
    except PageNotAnInteger:
        contents = paginator.page(1)
    except EmptyPage:
        contents = paginator.page(paginator.num_pages)

    return render(request, 'fileman/tags_detail.html', {
        'tag': tag,
        'contents': contents,
    })
